<?php
/**
 * WebViewer-Interface für FileMaker
 * Zeigt optimierte Routen mit Google Maps an
 */

// Session starten
session_start();

// Google Maps API Key - wird aus Session-Daten extrahiert
$apiKey = null;

// Session-ID aus URL
$sessionId = $_GET['session_id'] ?? '';

if (empty($sessionId)) {
    die('<h1>Fehler: Session-ID fehlt</h1><p>Bitte verwenden Sie den korrekten Link mit session_id Parameter.</p>');
}

// Session-Daten laden (mit dateibasiertem Fallback)
function loadFromSession($sessionId) {
    // Erst aus PHP-Session versuchen
    if (isset($_SESSION[$sessionId])) {
        $sessionData = $_SESSION[$sessionId];
        
        if (time() > $sessionData['expires']) {
            unset($_SESSION[$sessionId]);
            // Auch Datei löschen
            $sessionFile = __DIR__ . '/sessions/' . $sessionId . '.json';
            if (file_exists($sessionFile)) {
                unlink($sessionFile);
            }
            return null;
        }
        
        return $sessionData['data'];
    }
    
    // Fallback: Aus Datei laden
    $sessionFile = __DIR__ . '/sessions/' . $sessionId . '.json';
    if (!file_exists($sessionFile)) {
        return null;
    }
    
    $sessionContent = file_get_contents($sessionFile);
    if ($sessionContent === false) {
        return null;
    }
    
    $sessionData = json_decode($sessionContent, true);
    if (!$sessionData || !isset($sessionData['expires'])) {
        return null;
    }
    
    // Session-Ablauf prüfen
    if (time() > $sessionData['expires']) {
        unlink($sessionFile);
        return null;
    }
    
    // Wieder in PHP-Session laden (für Performance)
    $_SESSION[$sessionId] = $sessionData;
    
    return $sessionData['data'];
}

$sessionData = loadFromSession($sessionId);

if ($sessionData === null) {
    die('<h1>Fehler: Session nicht gefunden</h1><p>Die Session ist abgelaufen oder ungültig.</p>');
}

// Google API Key aus Session-Daten extrahieren
$apiKey = $sessionData['api_key'] ?? null;

if (!$apiKey) {
    die('<h1>Fehler: Google API Key nicht gefunden</h1><p>Der Google API Key ist in den Session-Daten nicht verfügbar.</p>');
}

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Route Planer - <?php echo htmlspecialchars($sessionId); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .container {
            max-width: 100%;
            margin: 0;
            padding: 0;
            height: 100vh;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 15px 20px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            text-align: center;
            position: relative;
            z-index: 1000;
        }
        
        .header h1 {
            color: #2c3e50;
            margin: 0;
            font-size: 1.8em;
        }
        
        .session-info {
            display: inline-block;
            margin-top: 10px;
            font-size: 0.9em;
            color: #6c757d;
        }
        
        .map-container {
            flex: 1;
            height: calc(100vh - 80px);
            overflow: hidden;
        }
        
        #map {
            height: 100%;
            width: 100%;
        }
        
        .loading {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(255, 255, 255, 0.95);
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
            text-align: center;
            z-index: 2000;
            display: none;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #007bff;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .legend {
            position: absolute;
            top: 10px;
            right: 10px;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 15px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            font-size: 12px;
            z-index: 1000;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            margin: 8px 0;
        }
        
        .legend-color {
            width: 16px;
            height: 16px;
            border-radius: 50%;
            margin-right: 8px;
            border: 2px solid white;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 2000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(5px);
        }
        
        .modal-content {
            background-color: white;
            margin: 5% auto;
            padding: 30px;
            border-radius: 15px;
            width: 90%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
        }
        
        .close {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }
        
        .close:hover {
            color: #000;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .control-grid {
                grid-template-columns: 1fr;
            }
            
            #map {
                height: 400px;
            }
            
            .session-info {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Routenplanung in Cluster</h1>
            <div class="session-info">
                Session: <?php echo htmlspecialchars($sessionId); ?> | 
                <?php echo $sessionData['total_contacts']; ?> Kontakte in 
                <?php echo $sessionData['total_clusters']; ?> Clustern
            </div>
        </div>
        
        <div class="map-container">
            <div id="map"></div>
            <div class="legend" id="legend"></div>
        </div>
        
        <div class="loading" id="loadingIndicator">
            <div class="spinner"></div>
            <p>Routen werden optimiert...</p>
        </div>
    </div>

    <script src="https://maps.googleapis.com/maps/api/js?key=<?php echo $apiKey; ?>"></script>
    <script>
        let map;
        let directionsService;
        let directionsRenderer = [];
        let directionsRendererByCluster = {};
        let clusterMarkers = {};
        let clusterBounds = {};
        let currentVisibleCluster = null;
        
        // Session-Daten von PHP
        const sessionId = '<?php echo $sessionId; ?>';
        const sessionData = <?php echo json_encode($sessionData, JSON_UNESCAPED_UNICODE); ?>;
        const departureTimeEpoch = sessionData.departure_time_epoch || null;
        const filemakerScriptName = (sessionData.filemaker_options && sessionData.filemaker_options.filemaker_script)
            ? sessionData.filemaker_options.filemaker_script
            : 'EventMaps';
        
        // Map initialisieren
        function initMap() {
            map = new google.maps.Map(document.getElementById('map'), {
                zoom: 8,
                center: { lat: 52.5, lng: 13.4 },
                mapTypeId: 'roadmap',
                styles: [
                    {
                        featureType: 'administrative',
                        elementType: 'geometry',
                        stylers: [{ visibility: 'off' }]
                    },
                    {
                        featureType: 'poi',
                        stylers: [{ visibility: 'off' }]
                    }
                ]
            });
            
            directionsService = new google.maps.DirectionsService();
            
            // Legende befüllen
            renderLegend();

            // Automatisch alle Routen optimieren und anzeigen
            initializeAndOptimizeRoutes();
        }
        
        // Alle Routen automatisch optimieren und anzeigen
        async function initializeAndOptimizeRoutes() {
            document.getElementById('loadingIndicator').style.display = 'block';
            
            try {
                console.log('Session Data:', sessionData);
                
                for (let clusterIndex = 0; clusterIndex < sessionData.clusters.length; clusterIndex++) {
                    await optimizeAndShowClusterRoute(clusterIndex);
                    
                    // Pause zwischen Clustern für Rate-Limiting
                    if (clusterIndex < sessionData.clusters.length - 1) {
                        await new Promise(resolve => setTimeout(resolve, 1000));
                    }
                }
                
            } catch (error) {
                console.error('Fehler bei der Route-Optimierung:', error);
            }
            
            document.getElementById('loadingIndicator').style.display = 'none';
        }
        
        // Cluster-Route optimieren und anzeigen
        async function optimizeAndShowClusterRoute(clusterIndex) {
            try {
                const cluster = sessionData.clusters[clusterIndex];
                console.log(`Optimiere Cluster ${clusterIndex + 1}:`, cluster);
                
                // Nur Kontakte mit gültigen Koordinaten verwenden
                const validContacts = cluster.filter(contact => 
                    contact.geocoding && 
                    contact.geocoding.success && 
                    contact.geocoding.lat && 
                    contact.geocoding.lng
                );
                
                if (validContacts.length < 2) {
                    console.log(`Cluster ${clusterIndex + 1} übersprungen: zu wenige gültige Koordinaten`);
                    return;
                }
                
                // Cluster-Farbe
                const clusterColor = getClusterColor(clusterIndex);
                
                // Start- und Endpunkt
                const origin = new google.maps.LatLng(
                    parseFloat(validContacts[0].geocoding.lat),
                    parseFloat(validContacts[0].geocoding.lng)
                );
                
                const destination = new google.maps.LatLng(
                    parseFloat(validContacts[validContacts.length - 1].geocoding.lat),
                    parseFloat(validContacts[validContacts.length - 1].geocoding.lng)
                );
                
                // Waypoints (falls mehr als 2 Kontakte)
                const waypoints = [];
                if (validContacts.length > 2) {
                    for (let i = 1; i < validContacts.length - 1; i++) {
                        waypoints.push({
                            location: new google.maps.LatLng(
                                parseFloat(validContacts[i].geocoding.lat),
                                parseFloat(validContacts[i].geocoding.lng)
                            ),
                            stopover: true
                        });
                    }
                }
                
                // Directions API Aufruf
                const directionsResult = await new Promise((resolve, reject) => {
                    const requestOptions = {
                        origin: origin,
                        destination: destination,
                        waypoints: waypoints,
                        optimizeWaypoints: true,
                        travelMode: google.maps.TravelMode.DRIVING,
                        unitSystem: google.maps.UnitSystem.METRIC,
                        region: 'DE'
                    };
                    if (departureTimeEpoch) {
                        requestOptions.drivingOptions = {
                            departureTime: new Date(departureTimeEpoch * 1000),
                            trafficModel: 'bestguess'
                        };
                    }
                    directionsService.route(requestOptions, (result, status) => {
                        if (status === 'OK') {
                            resolve(result);
                        } else {
                            reject(new Error(`Directions API Fehler: ${status}`));
                        }
                    });
                });
                
                // DirectionsRenderer für Route (ohne Marker)
                const renderer = new google.maps.DirectionsRenderer({
                    map: map,
                    directions: directionsResult,
                    routeIndex: 0,
                    suppressMarkers: true, // Eigene Marker verwenden
                    polylineOptions: {
                        strokeColor: clusterColor,
                        strokeWeight: 4,
                        strokeOpacity: 0.8
                    }
                });
                
                directionsRenderer.push(renderer);
                directionsRendererByCluster[clusterIndex] = renderer;
                if (directionsResult && directionsResult.routes && directionsResult.routes[0] && directionsResult.routes[0].bounds) {
                    clusterBounds[clusterIndex] = directionsResult.routes[0].bounds;
                }
                directionsRendererByCluster[clusterIndex] = renderer;
                if (directionsResult && directionsResult.routes && directionsResult.routes[0] && directionsResult.routes[0].bounds) {
                    clusterBounds[clusterIndex] = directionsResult.routes[0].bounds;
                }
                
                // Nummerierte Marker für alle Waypoints hinzufügen
                addNumberedMarkers(validContacts, clusterIndex, clusterColor);
                
                console.log(`Cluster ${clusterIndex + 1} erfolgreich optimiert`);
                
            } catch (error) {
                console.error(`Fehler bei Cluster ${clusterIndex + 1}:`, error);
            }
        }

        function showAllClusters() {
            currentVisibleCluster = null;
            Object.keys(directionsRendererByCluster).forEach(key => {
                const r = directionsRendererByCluster[key];
                if (r) r.setMap(map);
            });
            Object.keys(clusterMarkers).forEach(key => {
                (clusterMarkers[key] || []).forEach(m => m.setMap(map));
            });
        }

        function focusCluster(clusterIndex) {
            currentVisibleCluster = clusterIndex;
            Object.keys(directionsRendererByCluster).forEach(key => {
                const r = directionsRendererByCluster[key];
                if (r) r.setMap(parseInt(key,10) === clusterIndex ? map : null);
            });
            Object.keys(clusterMarkers).forEach(key => {
                (clusterMarkers[key] || []).forEach(m => m.setMap(parseInt(key,10) === clusterIndex ? map : null));
            });
            const b = clusterBounds[clusterIndex];
            if (b) {
                map.fitBounds(b);
            }
        }

        // Legende aus Session-Daten rendern
        function renderLegend() {
            const legend = document.getElementById('legend');
            if (!legend) return;
            const clusters = sessionData.clusters || [];
            let html = '<div style="font-weight:600;margin-bottom:6px;">Legende</div>';
            html += `<div class="legend-item" style="cursor:pointer;margin-bottom:6px;" data-cluster="all">
                        <span class="legend-color" style="background:#555"></span>
                        <span>Alle Cluster anzeigen</span>
                    </div>`;
            clusters.forEach((cluster, idx) => {
                const color = getClusterColor(idx);
                const count = Array.isArray(cluster) ? cluster.length : 0;
                html += `
                    <div class="legend-item" style="cursor:pointer;" data-cluster="${idx}">
                        <span class="legend-color" style="background:${color}"></span>
                        <span>Cluster ${idx + 1} (${count})</span>
                    </div>
                `;
            });
            legend.innerHTML = html;
            
            legend.querySelectorAll('.legend-item').forEach(el => {
                el.addEventListener('click', () => {
                    const val = el.getAttribute('data-cluster');
                    if (val === 'all') {
                        showAllClusters();
                    } else {
                        const idx = parseInt(val, 10);
                        focusCluster(idx);
                    }
                });
            });
        }
        
        // Nummerierte Marker für Cluster hinzufügen
        function addNumberedMarkers(contacts, clusterIndex, clusterColor) {
            if (!clusterMarkers[clusterIndex]) clusterMarkers[clusterIndex] = [];
            contacts.forEach((contact, contactIndex) => {
                const position = new google.maps.LatLng(
                    parseFloat(contact.geocoding.lat),
                    parseFloat(contact.geocoding.lng)
                );
                
                    const markerNumber = contactIndex + 1;
                    const isStart = contactIndex === 0;
                const isEnd = contactIndex === contacts.length - 1;
                
                // Marker-Icon je nach Position
                let markerIcon, labelText, labelColor;
                    
                    if (isStart) {
                        markerIcon = {
                            path: google.maps.SymbolPath.CIRCLE,
                        scale: 16,
                            fillColor: '#28a745', // Grün für Start
                            fillOpacity: 1,
                            strokeColor: '#ffffff',
                            strokeWeight: 3
                        };
                    labelText = 'S';
                    labelColor = '#ffffff';
                    } else if (isEnd) {
                        markerIcon = {
                            path: google.maps.SymbolPath.CIRCLE,
                        scale: 16,
                            fillColor: '#dc3545', // Rot für Ende
                            fillOpacity: 1,
                            strokeColor: '#ffffff',
                            strokeWeight: 3
                        };
                    labelText = 'Z';
                    labelColor = '#ffffff';
                    } else {
                        markerIcon = {
                            path: google.maps.SymbolPath.CIRCLE,
                        scale: 14,
                        fillColor: clusterColor,
                            fillOpacity: 0.9,
                            strokeColor: '#ffffff',
                            strokeWeight: 2
                        };
                    labelText = markerNumber.toString();
                    labelColor = '#ffffff';
                    }
                    
                    const marker = new google.maps.Marker({
                        position: position,
                        map: map,
                    title: `${markerNumber}. ${contact.full_name || contact.first + ' ' + contact.last}${isStart ? ' (START)' : isEnd ? ' (ZIEL)' : ''}`,
                        label: {
                        text: labelText,
                        color: labelColor,
                            fontWeight: 'bold',
                        fontSize: isStart || isEnd ? '16px' : '14px'
                    },
                    icon: markerIcon,
                    zIndex: isStart || isEnd ? 1000 : 100
                });
                
                // Info-Window mit Kontakt-Details. Hier können noch weitere übertragene Werte angezeigt werden. Des weiteren wird ein FileMaker Script aus dem InfoWindows per Button aufgerufen. Hier mit der id_contact
                const infoContent = `
                    <div style="max-width: 280px; font-family: Arial, sans-serif;">
                        <h4 style="color: ${isStart ? '#28a745' : isEnd ? '#dc3545' : clusterColor}; margin: 0 0 10px 0;">
                            ${isStart ? 'START' : isEnd ? 'ZIEL' : `${markerNumber}.`} ${contact.full_name || contact.first + ' ' + contact.last}
                        </h4>
                        <p style="margin: 5px 0;"><strong> Adresse:</strong><br>${contact.full_address}</p>
                        <p style="margin: 5px 0;"><strong> Cluster:</strong> ${clusterIndex + 1}</p>
                        <p style="margin: 5px 0; font-size: 12px; color: #666;">
                            <strong> Koordinaten:</strong> ${contact.geocoding.lat}, ${contact.geocoding.lng}
                        </p>
                        <div style="margin-top:10px;">
                            <button onclick="runFileMakerScript('${contact.id_contact}')" style="background:#007bff;color:#fff;border:none;border-radius:6px;padding:8px 12px;cursor:pointer;">
                                In FileMaker öffnen
                            </button>
                        </div>
                    </div>
                `;
                    
                    const infoWindow = new google.maps.InfoWindow({
                    content: infoContent
                    });
                    
                    marker.addListener('click', function() {
                        infoWindow.open(map, marker);
                    });
                    
                    clusterMarkers[clusterIndex].push(marker);
            });
        }
        
        // Cluster-Farbe basierend auf Index.
        function getClusterColor(clusterIndex) {
            const colors = [
                '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FECA57',
                '#FF9FF3', '#54A0FF', '#5F27CD', '#00D2D3', '#FF9F43',
                '#6C5CE7', '#A3CB38', '#FD79A8', '#E17055', '#74B9FF'
            ];
            return colors[clusterIndex % colors.length];
        }
        
        
        // Map initialisieren wenn Seite geladen
        google.maps.event.addDomListener(window, 'load', initMap);

        // FileMaker-Script starten
        function runFileMakerScript(idContact) {
            try {
                if (window.FileMaker && typeof window.FileMaker.PerformScriptWithOption === 'function') {
                    window.FileMaker.PerformScriptWithOption(filemakerScriptName, String(idContact), 5);
                    return;
                }
            } catch (e) {
                console.warn('FileMaker.PerformScriptWithOption nicht verfügbar:', e);
            }
            // Fallback: fmp:// URL öffnen (erfordert ggf. Anpassung der Datei/Host)
            const param = encodeURIComponent(String(idContact));
            const script = encodeURIComponent(filemakerScriptName);
            const url = `fmp://$/CurrentFile?script=${script}&param=${param}`;
            window.location.href = url;
        }
    </script>
</body>
</html>
